<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Stripe\StripeClient;
use Stripe\Exception\InvalidRequestException;
use Exception;

class Company extends Model
{
    use SoftDeletes;
    use HasFactory;

    const STATUS_ACTIVE = 1;
    const STATUS_COMING_SOON = 2;
    const STATUS_TEMP_CLOSE = 3;
    const STATUS_TEMP = 4;

    protected $guarded = [];

    // protected $fillable = [
    //     "id",
    //     "menu_type_id",
    //     "company_id",
    //     "day",
    //     "from",
    //     "to",
    // ];

    protected $casts = [
        'created_at' => 'datetime:Y-m-d',
        'meta' => 'array',
        'hours' => 'array',
        'tips' => 'array'
    ];

    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id', 'id')->select('name', 'id', 'type','phone_number');
    }

    public function users()
    {
        return $this->belongsTo(User::class, 'owner_id', 'id');
    }

    public function phone()
    {
        return $this->belongsTo(User::class, 'owner_id', 'id')->select('phone_number');
    }
    public function email()
    {
        return $this->belongsTo(User::class, 'owner_id', 'id')->select('email');
    }
    public function fullowner()
    {
        return $this->belongsTo(User::class, 'owner_id', 'id')->select('*');
    }

    public function type()
    {
        return $this->belongsTo(CompanyType::class, 'type_id')->select('id', 'name');
    }

    public function menuType()
    {
        return $this->belongsTo(CompanyMenuType::class, 'menu_type_id')
            ->select('id', 'name')
            ->with('menus');
    }

    public function promos()
    {
        return $this->hasMany(Promo::class)->with(['types']);
    }

    public function employeeArea()
    {
        return $this->hasOne(Employee::class)->with('area');
    }

    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    public function areas()
    {
        return $this->hasMany(Area::class);
    }

    public function discounts()
    {
        return $this->hasMany(Discount::class);
    }

    public function sizes()
    {
        return $this->hasMany(Size::class);
    }

    public function categories()
    {
        return $this->hasMany(Category::class)
            ->with(['menus']);
    }

    public function menus()
    {
        return $this->hasMany(Menu::class);
    }
    public function businesshours()
    {
        return $this->hasMany(Menu::class)->select('id','company_id','day','from','to');
    }

    public function menuTypes()
    {
        return $this->hasMany(MenuType::class);
    }

    public function options()
    {
        return $this->hasMany(Option::class);
    }

    public function optionGroup()
    {
        return $this->hasMany(OptionGroup::class);
    }

    public function reservations()
    {
        return $this->hasMany(Reservation::class, 'company_id');
    }

    public function file()
    {
        return $this->morphMany(File::class, 'fileable');
    }

    public function tables()
    {
        return $this->hasMany(Table::class);
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function userOrders()
    {
        return $this->hasMany(UserOrder::class);
    }

    public function claims()
    {
        return $this->hasMany(UserClaim::class);
    }


    public function companyclaim()
    {
        return $this->hasOne(Claim::class,'company_id', 'id');
    }
    public function regulargifts()
    {
        return $this->hasOne(RegularGiftSetting::class,'company_id', 'id');
    }

    public function banner()
    {
        return $this->morphMany(File::class, 'fileable')->where('fileable_type', Company::class)->latest();
    }

    public function scopeWithFilters($query)
    {
        return $query->when(!empty(request()->input('search')), function ($query) {
            $query->orders()->where('id', request()->input('search'))->where('id', '!=', null)->with('user');
        });
    }

    public function createTerminalLocation(){
        try {
            $stripe = new StripeClient(config('stripe.secret'));
            $address = [
                'line1' => $this->address_line_1,
                'city' => $this->city,
                'state' => $this->state,
                'country' => $this->country,
                'postal_code' => $this->postal_code,
            ];
            if($this->address_line_2) $address['line2'] = $this->address_line_2;
            $location = $stripe->terminal->locations->create([
                'display_name' => $this->name,
                'address' => $address
            ]);
            $this->terminal_location_id = $location->id;
            $this->save();
            return $location;
        } catch (InvalidRequestException $e) {
            return throw new Exception($e->getMessage());
        }
    }

    public function termianlLocation(){
        $stripe = new StripeClient(config('stripe.secret'));
        if($this->terminal_location_id){
            try{
                return $stripe->terminal->locations->retrieve( $this->terminal_location_id );
            } catch (InvalidRequestException $e) {
                return $this->createTerminalLocation();
            }
        }else{
            return $this->createTerminalLocation();
        }
    }

    public function getLatestdiscounts()
    {
        return $this->hasOne(Discount::class);
    }

    public function getLatestClaims()
    {
        return $this->hasOne(Claim::class,'company_id' ,'id');
    }

    public function getCompanyFollowers()
    {
        return $this->hasOne(Follow::class,'company_id','id');
    }

    public function getTotalFollowers()
    {
        return $this->hasMany(Follow::class,'company_id','id');
    }
    public static function boot()
    {
        parent::boot();

        self::creating(function($model){
            // ... code here

            $model->temporary_close   = $model->status == 3? 3:0;
            $model->coming_soon       = $model->status == 2? 2:0;
            $model->business_active   = $model->status == 1? 1:0;
        });
        self::updating(function($model){
            // ... code here
            $model->temporary_close   = $model->status == 3? 3:0;
            $model->coming_soon       = $model->status == 2? 2:0;
            $model->business_active   = $model->status == 1? 1:0;
        });

        self::updated(function($model){
            // ... code here
        });
    }

}
