<?php

namespace App\Nova\Actions;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Collection;
use Laravel\Nova\Actions\Action;
use Laravel\Nova\Fields\ActionFields;
use Laravel\Nova\Http\Requests\NovaRequest;
use Laravel\Nova\Fields\Text;
use Laravel\Nova\Fields\Boolean;
use Laravel\Nova\Fields\Select;
use Laravel\Nova\Fields\Number;
use Illuminate\Support\Facades\Validator;
use App\Helpers\Helper;
use App\Models\User;
use App\Models\UserPromo;
use App\Models\Transaction;
use App\Events\UserPromoUpdated;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class GiftCard extends Action
{
    use InteractsWithQueue, Queueable;

    public function handle(ActionFields $fields, Collection $models)
    {
        if($models->count() > 1) return Action::danger('You cannot create multiple Gift Cards at the same time using this action');
        $promoType = $models[0];

        if(!$promoType) return Action::danger('Promo Type is not found');

        if(!$promoType->promo?->isActive()) return Action::danger('Promo is not active');

        $phone = str_replace('+', '', $fields->phone);

        $user = User::by([
            'email' => $fields->email,
            'phone_number' => $phone,
            'password' => Hash::make(Str::random(60))
        ]);

        if($fields->pay_cash){
            $userPromo = UserPromo::create([
                'promo_id' => $promoType->promo_id,
                'promo_type_id' => $promoType->id,
                'user_id' => $user->id,
                'amount' => $fields->amount * 100,
                'taxes' => 0,
                'total' => $promoType->totalBy($fields->amount) * 100,
                'status' => UserPromo::STATUS_COMPLETED
            ]);
            UserPromo::updateInsummary($userPromo);
            event(new UserPromoUpdated($userPromo));
            return Action::redirect(config('app.url') . '/admin/resources/user-promos/' . $userPromo->id);
        }else{
            \Stripe\Stripe::setApiKey(config('stripe.secret'));

            $productData = [
                'name' => $promoType->title,
                'description' => $promoType->description
            ];

            $userPromo = UserPromo::create([
                'promo_id' => $promoType->promo_id,
                'promo_type_id' => $promoType->id,
                'user_id' => $user->id,
                'amount' => $fields->amount * 100,
                'taxes' => 0,
                'total' => $promoType->totalBy($fields->amount) * 100,
                'status' => UserPromo::STATUS_NEW
            ]);

            $sessionData = [
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'USD',
                        'product_data' => $productData,
                        'unit_amount' => (int) $promoType->totalBy($fields->amount) * 100
                    ],
                    'quantity' => 1,
                ]],
                'metadata' => [
                    'user_promo_id' => $userPromo->id
                ],
                'mode' => 'payment',
                'success_url' => config('app.url') . '/admin/resources/user-promos/' . $userPromo->id,
                'cancel_url' => config('app.url') . '/admin/resources/promo-types/' . $promoType->id
            ];
            if($fields->email) $sessionData['customer_email'] = $fields->email;
            $checkout_session = \Stripe\Checkout\Session::create($sessionData);

            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => Transaction::TYPE_GIFT_CARD,
                'company_id' => $userPromo->company_id,
                'payment_intent_id' => $checkout_session->payment_intent,
                'refund_promo_id' => $userPromo->id
            ]);

            $userPromo->transaction_id = $transaction->id;
            $userPromo->status = UserPromo::STATUS_PROCESSING;
            $userPromo->save();
            event(new UserPromoUpdated($userPromo));

            return Action::redirect($checkout_session->url);
        }
    }

    public function fields(NovaRequest $request)
    {

        $emailRules = [];
        $phoneRules = [];
        $phone = str_replace('+', '', $request->phone);

        if($phone && $request->code){
            $isPhoneValid = Helper::isPhoneValid($phone, $request->code);
            $emailRules = $isPhoneValid
                ? [ 'nullable', fn($a, $v, $f) => $request->email
                    && User::byPhone($phone)->exists()
                    && User::byPhone($phone)->first()->email != $request->email ? $f('Email does not fit user phone number') : null
                ] : [ 'required', 'email' ];
            $phoneRules = [ 'required',  fn($a, $v, $f) => !$isPhoneValid ? $f('Phone number is invalid') : null ];
        }else{
            $emailRules = [ 'required_without:phone', 'email' ];
            $phoneRules = [ 'nullable' ];
        }

        return [
            Number::make('Amount')->default(100)->rules( 'required', 'numeric', 'min:50', 'max:10000' ),
            Text::make('Email')
                ->rules( $emailRules ),
            Select::make('Code')
                ->options(
                    Helper::countryCodes()
                    ->map(fn($item) => [
                        'key' => $item[1],
                        'label' => $item[1]
                    ])
                    ->pluck('key', 'label')
                )
                ->rules('required')
                ->help('Phone Country Code')
                ->default('US'),
            Text::make('Phone')
                ->rules($phoneRules)
                ->help('Ex.: +12345678901'),
            Boolean::make('Pay Cash')
        ];
    }
}
