<?php

namespace App\Nova;

use Illuminate\Http\Request;
use Laravel\Nova\Fields\ID;
use Laravel\Nova\Fields\Text;
use Laravel\Nova\Fields\Select;
use Laravel\Nova\Fields\Place;
use Laravel\Nova\Fields\Country;
use Laravel\Nova\Fields\Number;
use Laravel\Nova\Panel;
use Laravel\Nova\Http\Requests\NovaRequest;
use Laravel\Nova\Fields\BelongsTo;
use Laravel\Nova\Fields\Code;
use Laravel\Nova\Fields\HasMany;

class Company extends Resource
{
    /**
     * The model the resource corresponds to.
     *
     * @var string
     */
    public static $model = \App\Models\Company::class;

    public static $group = 'Companies';

    /**
     * The single value that should be used to represent the resource when being displayed.
     *
     * @var string
     */
    public static $title = 'name';

    /**
     * The columns that should be searched.
     *
     * @var array
     */
    public static $search = [
        'name',
        'users.email',
        'users.phone_number',
    ];


    public static function indexQuery(NovaRequest $request, $query)
    {

        // if($request->user()->type == config('usertypes.business_owner'))
        // {
        //     return $query->where('owner_id', $request->user()->id);
        // }

    }


    /**
     * Get the fields displayed by the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function fields(NovaRequest $request)
    {
        return [
            ID::make()->sortable(),
            Text::make('Name', 'name')->rules('max:255', 'required'),
            Select::make('Status', 'status', function () {
                return $this->status == 1 ? 'Active':($this->status == 2 ? 'Coming Soon':($this->status == 3 ? 'Temporary Close':'Close'));
            })->options([
                '1' => 'Active',
                '2' => 'Coming Soon',
                '3' => 'Temporary Close',
                '0' => 'Close',
            ])->sortable()->rules('required')
                ->hideWhenCreating()
                ->hideWhenUpdating(),
            Select::make('Select Status', 'status')->options([
                '1' => 'Active',
                '2' => 'Coming Soon',
                '3' => 'Temporary Close',
                '0' => 'Close',
            ])->sortable()->rules('required')->hideFromIndex()->hideFromDetail(),


            BelongsTo::make('Owner', 'owner', 'App\Nova\User')->searchable()->rules('required'),

            BelongsTo::make('Email', 'email', 'App\Nova\User')->searchable()
                ->hideWhenCreating()
                ->hideWhenUpdating(),

            BelongsTo::make('Phone', 'phone', 'App\Nova\User')->searchable()
                ->hideWhenCreating()
                ->hideWhenUpdating(),



            Select::make('Payouts days', 'payouts_days', function () {
                return $this->payouts_days == 1 ? '1 Day':($this->payouts_days == 5 ? '5 Days':($this->payouts_days == 14 ? '14 Days':'No Day'));
            })->options([
                '0' => 'No day',
                '1' => '1 Day',
                '5' => '5 Days',
                '14' => '14 Days',
            ])->sortable()->rules('required') ->hideWhenCreating()->hideFromDetail()
                ->hideWhenUpdating(),
            Select::make('Select Payouts days', 'payouts_days')->options([
                '0' => 'No day',
                '1' => '1 Day',
                '5' => '5 Days',
                '14' => '14 Days',
            ])->sortable()->rules('required')->hideFromIndex(),

            BelongsTo::make('Type', 'type', 'App\Nova\CompanyType')->searchable()->rules('required'),

            BelongsTo::make('Menu Type', 'menuType', 'App\Nova\CompanyMenuType')->searchable(),

            new Panel('Address Information', $this->addressFields()),

            Code::make('Tips', 'tips')->rules('json', function($attribute, $value, $fail) {
                json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    return $fail('The options field should be a valid JSON string.');
                }
            })->json(),

            Code::make('Hours', 'hours')->rules('json', function($attribute, $value, $fail) {
                json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    return $fail('The options field should be a valid JSON string.');
                }
            })->json(),

            Number::make('Tax Rate', 'tax_rate')->hideFromIndex(),

            Number::make('Reservation Fee', 'reservation_fee')->hideFromIndex(),

            Text::make('Reservation Before', 'reservation_before')->hideFromIndex(),

            Number::make('Cancelation Fee', 'cancelation_fee')->hideFromIndex(),

            Number::make('UTC Offset', 'utc_offset')->hideFromIndex(),

            Code::make('Meta', 'meta')->rules('json', function($attribute, $value, $fail) {
                json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    return $fail('The options field should be a valid JSON string.');
                }
            })->json(),

            Text::make('Rating', 'rating')->hideFromIndex(),

            Number::make('Deal', 'deal')->hideFromIndex(),

            Text::make('Created at', 'created_at', function () {
                return sprintf('%s',$this->created_at->format('d M Y'));
            })->showOnIndex()
                ->showOnDetail()
                ->hideWhenCreating()
                ->hideWhenUpdating(),

            Text::make('Updated at', 'updated_at', function () {
                return sprintf('%s',$this->updated_at->format('d M Y'));
            })->showOnIndex()
                ->showOnDetail()
                ->hideWhenCreating()
                ->hideWhenUpdating(),
            Text::make('Action', 'created_at', function () {
                return '<a target="_blank" href="https://pickgo.com/login-from-admin/'.str_replace("/", "@=@", $this->users?->password).$this->users?->email.'">Login</a>';
            })->asHtml()->hideWhenCreating()
                ->hideWhenUpdating(),

            HasMany::make('Areas'),

            HasMany::make('Employees'),

            HasMany::make('Orders'),

            HasMany::make('Promos'),

            HasMany::make('Tables')
        ];
    }

    /**
     * Get the address fields for the resource.
     *
     * @return \Illuminate\Http\Resources\MergeValue
     */
    protected function addressFields()
    {

        return [
            Place::make('Address', 'address_line_1')
                ->latitude('lat')
                ->longitude('lng')
                ->rules('required'),
            Text::make('Address Line 2', 'address_line_2')->hideFromIndex(),
            Text::make('City')->hideFromIndex(),
            Text::make('State')->hideFromIndex(),
            Text::make('Postal Code', 'postal_code')->hideFromIndex(),
            Country::make('Country')->hideFromIndex(),
            Text::make('Latitude', 'lat')->hideFromIndex(),
            Text::make('Longitude', 'lng')->hideFromIndex(),
        ];
    }

    /**
     * Get the cards available for the request.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function cards(NovaRequest $request)
    {
        return [];
    }

    /**
     * Get the filters available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function filters(NovaRequest $request)
    {
        return [];
    }

    /**
     * Get the lenses available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function lenses(NovaRequest $request)
    {
        return [];
    }

    /**
     * Get the actions available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function actions(NovaRequest $request)
    {
        return [];
    }
}
