<?php

namespace App\Nova;

use Illuminate\Http\Request;
use Illuminate\Validation\Rules;
use Laravel\Nova\Fields\Date;
use Laravel\Nova\Fields\Gravatar;
use Laravel\Nova\Fields\ID;
use Laravel\Nova\Fields\Password;
use Laravel\Nova\Fields\Text;
use Laravel\Nova\Fields\Select;
use Laravel\Nova\Fields\DateTime;
use Laravel\Nova\Fields\HasOne;
use Laravel\Nova\Http\Requests\NovaRequest;
use Laravel\Nova\Fields\HasMany;
use App\Models\User as UserModel;
use Arr;

class User extends Resource
{
    /**
     * The model the resource corresponds to.
     *
     * @var string
     */
    public static $model = \App\Models\User::class;

    /**
     * The single value that should be used to represent the resource when being displayed.
     *
     * @var string
     */
    public function title(){
        return $this->anyName();
    }

    /**
     * The columns that should be searched.
     *
     * @var array
     */
    public static $search = [
        'id', 'name', 'email',
    ];

   
    /**
     * Build an "index" query for the given resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    

    public static function indexQuery(NovaRequest $request, $query)
    {
        
        if($request->user()->type == config('usertypes.admin')) 
        {
            return $query->where('type', 0)->orWhere('type',1)->orWhere('id', $request->user()->id);
        }

        if($request->user()->type == config('usertypes.business_owner')) 
        {
            return $query->Where('type', 0)->orWhere('type',1);
        } 

        if($request->user()->type == config('usertypes.customer')) 
        {
            return $query->Where('id', $request->user()->id);
        } 
        
    }
  
    /**
     * Get the fields displayed by the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function fields(NovaRequest $request)
    {
        return [
            ID::make()->sortable(),

            Gravatar::make()->maxWidth(50),
            
            Text::make('Name')
                ->sortable()
                ->rules('required', 'max:255'),

            Text::make('Email')
                ->sortable()
                ->rules('required', 'email', 'max:254')
                ->creationRules('unique:users,email')
                ->updateRules('unique:users,email,{{resourceId}}'),

            Text::make('Phone Number')
                ->sortable(),

            Password::make('Password')
                ->onlyOnForms()
                ->rules('nullable', Rules\Password::defaults()),

            Select::make('User Type', 'type')->options(UserModel::types())->onlyOnForms()->rules('required'),

            Text::make('User Type', 'type')
                ->resolveUsing(fn() => Arr::get(UserModel::types(), $this->type))
                ->showOnIndex()
                ->showOnDetail()
                ->hideWhenUpdating()
                ->hideWhenCreating()
                ->sortable(),

            Select::make('Medical Status', 'medical_status', function () {
                return $this->medical_status;
            })->sortable()->rules('required')
                ->hideWhenCreating()
                ->hideFromDetail()
                ->hideWhenUpdating(),
            Select::make('Medical Status', 'medical_status')->options([
                'Pending' => 'Pending',
                'Approved' => 'Approved',
                'Rejected' => 'Rejected',
            ])->sortable()
             ->rules('required')
             ->hideFromIndex(),
            Text::make('Medical Card No','medical_card_no'),
            Date::make('Date Of Birth','date_of_birth'),
            Text::make('Medical Card State','meddical_card_state'),
            Text::make('Caregiver Id Number','caregiver_id_number'),
            Date::make('Date Of Birth','medical_card_expiration'),

            Text::make('Medical Photo', 'photo', function () { //medical_status
                return "<img src='{$this->photo}' alt='{$this->name}' />";
            })->asHtml()
                ->hideWhenCreating()
                ->hideWhenUpdating(),

            Text::make('Photo ID', 'photo_id', function () {
                return "<img src='{$this->photo_id}' alt='{$this->name}' />";
            })->asHtml()
                ->hideWhenCreating()
                ->hideWhenUpdating(),

            DateTime::make('Created at'),

            DateTime::make('Updated at'),

            DateTime::make('Email Verified at'),

            HasMany::make('Companies'),

            HasMany::make('User Promos'),
            
        ];
    }

    /**
     * Get the cards available for the request.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function cards(NovaRequest $request)
    {
        return [];
    }

    /**
     * Get the filters available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function filters(NovaRequest $request)
    {
        return [
            new Filters\UserType,
        ];
    }

    /**
     * Get the lenses available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function lenses(NovaRequest $request)
    {
        return [];
    }

    /**
     * Get the actions available for the resource.
     *
     * @param  \Laravel\Nova\Http\Requests\NovaRequest  $request
     * @return array
     */
    public function actions(NovaRequest $request)
    {
        return [];
    }
}
